/*
 * pl_mpeg - MPEG1 Video Decoder / Player - SDL 1.2 port (video only)
 * Original SDL2 version by Dominic Szablewski - https://phoboslab.org
 * Ported to SDL 1.2 and audio removed
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#define PL_MPEG_IMPLEMENTATION
#include "pl_mpeg.h"

#include <SDL/SDL.h>

typedef struct {
    plm_t *plm;
    double last_time;
    int wants_to_quit;

    SDL_Surface *screen;
    SDL_Overlay *overlay;
} app_t;

app_t * app_create(const char *filename);
void app_destroy(app_t *self);
void app_update(app_t *self);

void app_on_video(plm_t *plm, plm_frame_t *frame, void *user);

int main(int argc, char *argv[]) {
    if (argc < 2) {
        fprintf(stderr, "Usage: %s <file.mpg>\n", argv[0]);
        return 1;
    }

    if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_TIMER) < 0) {
        fprintf(stderr, "SDL_Init failed: %s\n", SDL_GetError());
        return 1;
    }

    app_t *app = app_create(argv[1]);
    if (!app) {
        SDL_Quit();
        return 1;
    }

    while (!app->wants_to_quit) {
        app_update(app);
        SDL_Delay(2);  // gentle cpu usage
    }

    app_destroy(app);
    SDL_Quit();
    return 0;
}

app_t * app_create(const char *filename) {
    app_t *self = (app_t *)calloc(1, sizeof(app_t));
    if (!self) return NULL;

    self->plm = plm_create_with_filename(filename);
    if (!self->plm) {
        fprintf(stderr, "Couldn't open %s\n", filename);
        free(self);
        return NULL;
    }

    if (!plm_probe(self->plm, 5000 * 1024)) {
        fprintf(stderr, "No MPEG video stream found in %s\n", filename);
        plm_destroy(self->plm);
        free(self);
        return NULL;
    }

    int w = plm_get_width(self->plm);
    int h = plm_get_height(self->plm);

    printf("Opened %s - %dx%d @ %.2f fps (video only)\n",
           filename, w, h, plm_get_framerate(self->plm));

    // Video setup - SDL 1.2 style
    Uint32 flags = SDL_ANYFORMAT | SDL_DOUBLEBUF | SDL_RESIZABLE;
    self->screen = SDL_SetVideoMode(w, h, 0, flags);
    if (!self->screen) {
        fprintf(stderr, "SDL_SetVideoMode failed: %s\n", SDL_GetError());
        goto fail;
    }

    self->overlay = SDL_CreateYUVOverlay(w, h, SDL_YV12_OVERLAY, self->screen);
    if (!self->overlay) {
        fprintf(stderr, "SDL_CreateYUVOverlay failed: %s\n", SDL_GetError());
        goto fail;
    }

    SDL_WM_SetCaption("pl_mpeg - SDL 1.2 player (video only)", "pl_mpeg");

    plm_set_loop(self->plm, TRUE);
    plm_set_video_decode_callback(self->plm, app_on_video, self);

    return self;

fail:
    app_destroy(self);
    return NULL;
}

void app_destroy(app_t *self) {
    if (!self) return;

    if (self->overlay) {
        SDL_FreeYUVOverlay(self->overlay);
    }

    if (self->plm) {
        plm_destroy(self->plm);
    }

    free(self);
}

void app_update(app_t *self) {
    double seek_to = -1.0;

    SDL_Event ev;
    while (SDL_PollEvent(&ev)) {
        if (ev.type == SDL_QUIT ||
            (ev.type == SDL_KEYDOWN && ev.key.keysym.sym == SDLK_ESCAPE)) {
            self->wants_to_quit = 1;
        }
        else if (ev.type == SDL_KEYDOWN) {
            if (ev.key.keysym.sym == SDLK_RIGHT) {
                seek_to = plm_get_time(self->plm) + 3.0;
            }
            else if (ev.key.keysym.sym == SDLK_LEFT) {
                seek_to = plm_get_time(self->plm) - 3.0;
            }
            else if (ev.key.keysym.sym == SDLK_f) {
                // Toggle fullscreen
                Uint32 flags = self->screen->flags ^ SDL_FULLSCREEN;
                SDL_Surface *newscreen = SDL_SetVideoMode(
                    self->screen->w, self->screen->h, 0,
                    flags | SDL_ANYFORMAT | SDL_DOUBLEBUF);
                if (newscreen) {
                    self->screen = newscreen;
                    SDL_FreeYUVOverlay(self->overlay);
                    self->overlay = SDL_CreateYUVOverlay(
                        plm_get_width(self->plm), plm_get_height(self->plm),
                        SDL_YV12_OVERLAY, self->screen);
                }
            }
        }
        else if (ev.type == SDL_VIDEORESIZE) {
            // Re-create overlay on resize (keeps native resolution)
            SDL_FreeYUVOverlay(self->overlay);
            self->overlay = SDL_CreateYUVOverlay(
                plm_get_width(self->plm), plm_get_height(self->plm),
                SDL_YV12_OVERLAY, self->screen);
        }
    }

    // Mouse seek (left button drag on progress bar feel)
    if (SDL_GetMouseState(NULL, NULL) & SDL_BUTTON(1)) {
        int mx, my;
        SDL_GetMouseState(&mx, &my);
        seek_to = plm_get_duration(self->plm) * ((float)mx / self->screen->w);
    }

    double now = SDL_GetTicks() / 1000.0;
    double dt = now - self->last_time;
    self->last_time = now;

    // Cap delta time to avoid big jumps after pausing/minimizing
    if (dt > 1.0 / 30.0) dt = 1.0 / 30.0;

    if (seek_to >= 0.0) {
        plm_seek(self->plm, seek_to, 0);
    }
    else {
        plm_decode(self->plm, dt);
    }

    if (plm_has_ended(self->plm)) {
        self->wants_to_quit = 1;
    }
}

void app_on_video(plm_t *plm, plm_frame_t *frame, void *user) {
    app_t *self = (app_t *)user;

    if (SDL_LockYUVOverlay(self->overlay) < 0) {
        return;
    }

    // Y plane
    int y;
    Uint8 *dst = self->overlay->pixels[0];
    Uint8 *src = frame->y.data;
    for (y = 0; y < frame->height; y++) {
        memcpy(dst, src, frame->y.width);
        dst += self->overlay->pitches[0];
        src += frame->y.width;
    }

    // Cb (U) plane
    dst = self->overlay->pixels[1];
    src = frame->cb.data;
    for (y = 0; y < frame->height / 2; y++) {
        memcpy(dst, src, frame->cb.width);
        dst += self->overlay->pitches[1];
        src += frame->cb.width;
    }

    // Cr (V) plane
    dst = self->overlay->pixels[2];
    src = frame->cr.data;
    for (y = 0; y < frame->height / 2; y++) {
        memcpy(dst, src, frame->cr.width);
        dst += self->overlay->pitches[2];
        src += frame->cr.width;
    }

    SDL_UnlockYUVOverlay(self->overlay);

    SDL_Rect rect = {0, 0, self->screen->w, self->screen->h};
    SDL_DisplayYUVOverlay(self->overlay, &rect);
}



